<?php // phpcs:ignore
/**
 * Product submenu class
 *
 * @category   Klasse_Theme
 * @package    KT
 * @subpackage KT/includes/woocommerce
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.1.74
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
	exit;
}

if (!class_exists('KT_Product_Submenu')) {
	/**
	 * Product submenu class
	 *
	 * @category    Klasse_Theme
	 * @package     KT
	 * @subpackage  KT/includes/woocommerce
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.1.74
	 */
	class KT_Product_Submenu
	{

		/**
		 * Construct the product submenu.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.74
		 * 
		 * Todo, transients should be emptied when updating the menu and when changing the product categorie order.
		 * Todo, this should be optimized for speed.
		 */
		public function __construct()
		{
			if (get_theme_mod('kls_product_menu_toggle', 0) == 1) {
				add_filter('wp_nav_menu', array($this, 'replacePrimaryMenu'), 10, 2);
			}

			add_filter('wp_nav_menu_args', array($this, 'setup_custom_walker'));

			add_action('save_post', array($this, 'clearMenuTransient'));
		}

		/**
		 * Add the product submenu items into the submenu.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.74
		 */

		public function setup_custom_walker($args)
		{
			if (isset($args['theme_location']) && $args['theme_location'] == 'primary') {
				$args['walker'] = new Kls_Custom_Walker_Nav_Menu();
			}

			return $args;
		}

		/**
		 * Create the product submenu.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.74
		 */
		public function createProductSubmenu($submenu = false, $only_current_category = false, $category_id = false)
		{
			if ($only_current_category) {
				$transient_name = 'kls_menu_transient_' . $category_id;
			} else {
				$transient_name = 'kls_menu_transient';
			}

			$menu_html = get_transient($transient_name);

			if (!$menu_html) {
				$hide_empty_categories = get_theme_mod('kls_product_menu_empty_categories_toggle', 1) == 1 ? true : false;
				$menu_type = get_theme_mod('kls_product_menu_submenu_version', 'wide-menu');

				if ($menu_type == 'basic-menu') {
					$menu_type_classes = '';
				} else {
					$menu_type_classes = 'mega-menu mega-product-menu';
				}

				if ($only_current_category) {
					$categories[] = get_term($category_id);
				} else {
					$args = array(
						'taxonomy' => 'product_cat',
						'hide_empty' => $hide_empty_categories,
						'parent' => 0,
					);
					$categories = get_terms($args);
				}

				$exclude = explode(',', get_theme_mod('kls_product_menu_exclude_ids'));
				if ($submenu) {
					$html = '<ul class="sub-menu">';
				} else {
					$html = '<ul class="menu">';
				}

				$html = apply_filters('kls_before_product_menu', $html);

				foreach ($categories as $category) {
					if (!in_array($category->term_id, $exclude)) {
						$sub_args = array(
							'taxonomy' => 'product_cat',
							'hide_empty' => $hide_empty_categories,
							'parent' => $category->term_id,
						);
						$sub_categories = get_terms($sub_args);

						if (!empty($sub_categories)) {
							if (!$only_current_category) {
								$html .= '<li class="menu-item menu-item-type-post_type menu-item-object-page menu-item-has-children ' . $menu_type_classes . '"><a href="' . get_term_link($category->term_id, 'product_cat') . '">' . $category->name . '</a>';

								$html .= '<ul class="sub-menu">';
							}
							foreach ($sub_categories as $sub_category) {
								$html .= $this->getSubCategory($sub_category->term_id, $exclude);
							}

							if (get_theme_mod('kls_product_menu_showcase_toggle') == 1) {
								$html .= '<li class="product-showcase-item"><div class="product-menu-showcase" style="display:none;"></div></li>';
							}
							if (!$only_current_category) {
								$html .= '</ul>';
								$html .= '</li>';
							}
						} else {
							$product_args = array(
								'post_type' => 'product',
								'posts_per_page' => -1,
								'tax_query' => array(
									array(
										'taxonomy' => 'product_cat',
										'field' => 'term_id',
										'terms' => $category->term_id,
									),
								),
								'order' => 'ASC',
								'orderby' => 'title',
							);
							$products = new WP_Query($product_args);

							if ($products->have_posts()) {
								$html .= '<li class="menu-item menu-item-type-post_type menu-item-object-page menu-item-has-children ' . $menu_type_classes . '"><a href="' . get_term_link($category->term_id, 'product_cat') . '">' . $category->name . '</a>';

								$html .= '<ul class="sub-menu">';

								$html .= $this->getSubCategory($category->term_id, $exclude);

								if (get_theme_mod('kls_product_menu_showcase_toggle') == 1) {
									$html .= '<li class="product-showcase-item"><div class="product-menu-showcase" style="display:none;"></div></li>';
								}
								$html .= '</ul>';

								$html .= '</li>';
							} else {
								$html .= '<li class="menu-item menu-item-type-post_type menu-item-object-page"><a href="' . get_term_link($category->term_id, 'product_cat') . '">' . $category->name . '</a></li>';
							}
						}
					}
				}

				$html = apply_filters('kls_after_product_menu', $html);

				wp_reset_query();

				$menu_html = $html . '</ul>';

				set_transient($transient_name, $menu_html, WEEK_IN_SECONDS);
			}

			return $menu_html;
		}

		/**
		 * Get the sub category html.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.74
		 */
		public function getSubCategory($sub_category, $exclude = null)
		{
			$max_products = get_theme_mod('kls_product_menu_max_products', '0') == '0' ? '99999' : get_theme_mod('kls_product_menu_max_products');
			$html = '';

			if (!in_array($sub_category, $exclude)) {
				$product_args = array(
					'post_type' => 'product',
					'posts_per_page' => -1,
					'tax_query' => array(
						array(
							'taxonomy' => 'product_cat',
							'field' => 'term_id',
							'terms' => $sub_category,
						),
					),
					'order' => 'ASC',
					'orderby' => 'title',
				);
				$products = new WP_Query($product_args);

				if ($products->have_posts()) {
					$html = '<li class="menu-item menu-item-type-post_type menu-item-object-page menu-item-has-children"><a href="' . get_term_link($sub_category, 'product_cat') . '">' . get_term($sub_category)->name . '</a>';

					$html .= '<ul class="sub-menu">';
					$i = 0;
					while ($products->have_posts()) {
						if ($i > $max_products) {
							$html .= '<li class="menu-item product-item menu-item-type-post_type menu-item-object-page"><a href="' . get_term_link($sub_category, 'product_cat') . '">' . __('View more', 'klasse-theme') . '...</a></li>';
							break;
						}
						$i++;
						$products->the_post();

						$product_instance = wc_get_product(get_the_id());
						$product_short_description = $product_instance->get_short_description();

						if (strlen($product_short_description) > 50) {
							$product_short_description = substr($product_short_description, 0, 47) . '...';
						}

						$product_name = get_the_title();
						$product_thumbnail = get_post_thumbnail_id(get_the_id());

						$product_24 = get_post_meta(get_the_id(), '24hour_shipping');
						$product_24_html = isset($product_24) && !empty($product_24) && $product_24[0] == 'yes' ? '<span class="always-shipping">24u</span>' : '';
						$product_24 = isset($product_24) && !empty($product_24) && $product_24[0] == 'yes' ? '1' : '';

						if ($product_thumbnail == null) {
							$product_image = '';
						} else {
							$product_image = wp_get_attachment_image_src($product_thumbnail, 'thumbnail')[0];
						}

						$html .= '<li class="menu-item product-item menu-item-type-post_type menu-item-object-page" data-product-id="' . get_the_id() . '" data-product-image="' . $product_image . '" data-product-description="' . strip_tags($product_short_description) . '" data-product-name="' . $product_name . '" data-product-24="' . $product_24 . '"><a href="' . get_permalink() . '">' . $product_name . $product_24_html . '</a></li>';
					}
					$i = 0;
					wp_reset_query();

					$html .= '</ul>';
					$html .= '</li>';
				} else {
					$html = '<li class="menu-item menu-item-type-post_type menu-item-object-page"><a href="' . get_term_link($sub_category, 'product_cat') . '">' . get_term($sub_category)->name . '</a></li>';
				}
			}
			return $html;
		}

		/**
		 * Replace the primary menu if it's toggled in the customizer.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.74
		 */
		public function replacePrimaryMenu($menu, $args)
		{
			if ($args->theme_location == 'primary') {
				$menu = $this->createProductSubmenu();
			}
			return $menu;
		}

		/**
		 * Clear the menu transient if the posts get saved.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 * 
		 * @since 0.1.78
		 */
		public function clearMenuTransient()
		{
			delete_transient('kls_menu_transient');
			$this->delete_transients_with_prefix('kls_menu_transient');
		}

		private function delete_transients_with_prefix($prefix)
		{
			foreach ($this->get_transient_keys_with_prefix($prefix) as $key) {
				delete_transient($key);
			}
		}

		private function get_transient_keys_with_prefix($prefix)
		{
			global $wpdb;

			$prefix = $wpdb->esc_like('_transient_' . $prefix);
			$sql = "SELECT `option_name` FROM $wpdb->options WHERE `option_name` LIKE '%s'";
			$keys = $wpdb->get_results($wpdb->prepare($sql, $prefix . '%'), ARRAY_A);

			if (is_wp_error($keys)) {
				return [];
			}

			return array_map(function ($key) {
				return substr($key['option_name'], strlen('_transient_'));
			}, $keys);
		}
	}

	new KT_Product_Submenu();
}

class Kls_Custom_Walker_Nav_Menu extends Walker_Nav_Menu
{
	function start_el(&$output, $item, $depth = 0, $args = null, $id = 0)
	{
		$checkbox_value = get_post_meta($item->ID, '_kls-show-product-menu', true);
		$only_category_products = get_post_meta($item->ID, '_kls-only-show-category-products', true);

		if ($checkbox_value == '1') {
			$item->classes[] = 'menu-item-has-children';

			if ($only_category_products == '1') {
				$item->classes[] = 'mega-menu mega-product-menu';
				$category_id = $item->object_id;
			} else {
				$item->classes[] = get_theme_mod('kls_product_menu_submenu_version', 'wide-menu');
			}
		}

		parent::start_el($output, $item, $depth, $args, $id);

		if ($checkbox_value == '1') {
			$category_id = ($only_category_products == '1') ? $item->object_id : '';
			$kt_submenu = new KT_Product_Submenu();
			$submenu_html = $kt_submenu->createProductSubmenu(true, $only_category_products, $category_id);
			$output .= $submenu_html;
		}
	}
}
