<?php
/**
 * Register the custom Divi button module
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.1.14
 *
 * [Dev] Todo, Add Anytime fitness toggles
 * Todo, add toggle to make the button inline-block.
 */
class KLS_ButtonModule extends et_builder_Module {

	public $slug       = 'kls_button';
	public $vb_support = 'partial';

	/**
	 * Initialize the module class
	 *
	 * @author Joeri van der Stek <joeri@klassebv.nl>
	 *
	 * @since 0.1.14
	 */
	public function init() {
		$this->name             = esc_html__( 'Theme button', 'klasse-theme' );
		$this->fb_support       = true;
		$this->main_css_element = '%%order_class%%';

		$this->settings_modal_toggles = array(
			'advanced' => array(
				'toggles' => array(
					'alignment' => esc_html__( 'Alignment', 'et_builder' ),
					'text'      => array(
						'title'    => __( 'Text' ),
						'priority' => 49,
					),
				),
			),
		);

		$this->advanced_fields = array(
			'background'         => false,
			'box_shadow'         => false,
			'max_width'          => false,
			'borders'            => false,
			'fonts'              => false,
			'text'               => false,
			'button'             => false,
			'link_options'       => false,
			'filters'            => false,
			'transform'          => false,
			'animation'          => true, // Todo, animations should be set on the wrapper. Not on the a itself.
			'display_conditions' => false,
			'sticky'             => false,
			'scroll_effects'     => false,
			'margin_padding'     => array(
				'css' => array(
					'padding'   => "{$this->main_css_element}_wrapper {$this->main_css_element}, {$this->main_css_element}_wrapper {$this->main_css_element}:hover",
					'margin'    => "{$this->main_css_element}_wrapper {$this->main_css_element}",
					'important' => 'all',
				),
			),
		);
	}

	/**
	 * Get's the module fields.
	 *
	 * @author Joeri van der Stek <joeri@klassebv.nl>
	 *
	 * @since 0.1.14
	 */
	public function get_fields() {
		$fields = array(
			'button_text'       => array(
				'label'           => esc_html__( 'Button text', 'klasse-theme' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'description'     => esc_html__( 'Input your desired text here.', 'klasse-theme' ),
				'toggle_slug'     => 'main_content',
			),
			'button_styling'    => array(
				'label'           => esc_html__( 'Button styling', 'klasse-theme' ),
				'type'            => 'select',
				'option_category' => 'basic_option',
				'description'     => esc_html__( 'Select the styling of the button. This styling is defined in the customizer.', 'klasse-theme' ),
				'toggle_slug'     => 'main_content',
				'options'         => $GLOBALS['kls_buttons'],
			),
			'button_full_width' => array(
				'label'           => esc_html__( 'Button full width', 'klasse-theme' ),
				'type'            => 'yes_no_button',
				'option_category' => 'basic_option',
				'description'     => esc_html__( 'Select if the button has to be full width or not.', 'klasse-theme' ),
				'toggle_slug'     => 'main_content',
				'options'         => array(
					'off' => esc_html__( 'No', 'klasse-theme' ),
					'on'  => esc_html__( 'Yes', 'klasse-theme' ),
				),
			),
			'button_inline' => array(
				'label' => esc_html__('Button inline', 'klasse-theme'),
				'type' => 'yes_no_button',
				'option_category' => 'basic_option',
				'description' => esc_html__('Select if the button has to be inline.', 'klasse-theme'),
				'toggle_slug' => 'main_content',
				'options' => array(
					'off' => esc_html__('No', 'klasse-theme'),
					'on' => esc_html__('Yes', 'klasse-theme'),
				),
			),
			'button_url'        => array(
				'label'           => esc_html__( 'Button Link URL', 'klasse-theme' ),
				'type'            => 'text',
				'option_category' => 'basic_option',
				'description'     => esc_html__( 'Input the destination URL for your button.', 'klasse-theme' ),
				'toggle_slug'     => 'link',
				'dynamic_content' => 'url',
			),
			'url_new_window'    => array(
				'label'            => esc_html__( 'Button Link Target', 'klasse-theme' ),
				'type'             => 'select',
				'option_category'  => 'configuration',
				'options'          => array(
					'off' => esc_html__( 'In the same window', 'klasse-theme' ),
					'on'  => esc_html__( 'In the new tab', 'klasse-theme' ),
				),
				'toggle_slug'      => 'link',
				'description'      => esc_html__( 'Here you can choose whether or not your link opens in a new window', 'klasse-theme' ),
				'default_on_front' => 'off',
			),
			'button_alignment'  => array(
				'label'           => esc_html__( 'Button Alignment', 'klasse-theme' ),
				'description'     => esc_html__( 'Align your button to the left, right or center of the module.', 'klasse-theme' ),
				'type'            => 'text_align',
				'option_category' => 'configuration',
				'options'         => et_builder_get_text_orientation_options( array( 'justified' ) ),
				'tab_slug'        => 'advanced',
				'toggle_slug'     => 'alignment',
				'description'     => esc_html__( 'Here you can define the alignment of Button', 'klasse-theme' ),
				'mobile_options'  => true,
			),
		);

		return $fields;
	}

	/**
	 * Get button alignment.
	 */
	public function get_button_alignment( $device = 'desktop' ) {
		$suffix           = 'desktop' !== $device ? "_{$device}" : '';
		$text_orientation = isset( $this->props[ "button_alignment{$suffix}" ] ) ? $this->props[ "button_alignment{$suffix}" ] : '';

		return et_pb_get_alignment( $text_orientation );
	}

	/**
	 * Render the module output.
	 *
	 * @param  array  $attrs       List of attributes.
	 * @param  string $content     Content being processed.
	 * @param  string $render_slug Slug of module that is used for rendering output.
	 *
	 * @author Joeri van der Stek <joeri@klassebv.nl>
	 *
	 * @since 0.1.14
	 *
	 * @return string
	 */
	public function render( $unprocessed_props, $content, $render_slug ) {
		$styling    = ( $this->props['button_styling'] ? $this->props['button_styling'] : 'primary' );
		$url        = $this->props['button_url'];
		$text       = $this->props['button_text'];
		$new_tab    = ( $this->props['url_new_window'] == 'on' ? 'target="_blank"' : '' );
		$full_width = ( $this->props['button_full_width'] == 'on' ? 'theme-button-full-width' : '' );
		$inline = ($this->props['button_inline'] == 'on' ? 'button_module_inline' : '');

		$button_alignment              = $this->get_button_alignment();
		$is_button_aligment_responsive = et_pb_responsive_options()->is_responsive_enabled( $this->props, 'button_alignment' );
		$button_alignment_tablet       = $is_button_aligment_responsive ? $this->get_button_alignment( 'tablet' ) : '';
		$button_alignment_phone        = $is_button_aligment_responsive ? $this->get_button_alignment( 'phone' ) : '';

		// Button Alignment.
		$button_alignments = array();
		if ( ! empty( $button_alignment ) ) {
			array_push( $button_alignments, sprintf( 'button_alignment_%1$s', esc_attr( $button_alignment ) ) );
		}

		if ( ! empty( $button_alignment_tablet ) ) {
			array_push( $button_alignments, sprintf( 'button_alignment_tablet_%1$s', esc_attr( $button_alignment_tablet ) ) );
		}

		if ( ! empty( $button_alignment_phone ) ) {
			array_push( $button_alignments, sprintf( 'button_alignment_phone_%1$s', esc_attr( $button_alignment_phone ) ) );
		}

		$button_alignment_classes = join( ' ', $button_alignments );

		return sprintf(
			'<div class="button_module_wrapper %7$s_wrapper %6$s %10$s"><a class="theme-button theme-button-%1$s %5$s %7$s %8$s %9$s" href="%2$s" %3$s >%4$s</a></div>',
			$styling,
			$url,
			$new_tab,
			$text,
			$full_width,
			esc_attr( $button_alignment_classes ),
			esc_attr( ET_Builder_Element::get_module_order_class( $this->slug ) ),
			$this->module_id( $render_slug ),
			$this->module_classname( $render_slug ),
			$inline
		);
	}
}

new KLS_ButtonModule();
