<?php // phpcs:ignore
/**
 * Parameter class
 *
 * @category   Klasse_Plugin
 * @package    KP
 * @subpackage KP/public/includes/tracking
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.1.16
 */

// Exit if accessed directly.
if (!defined('ABSPATH')) {
	exit;
}

if (!class_exists('KP_Tracking_Parameter')) {
	/**
	 * Parameter class
	 *
	 * @category    Klasse_Plugin
	 * @package     KP
	 * @subpackage  KP/public/includes/tracking
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.1.16
	 */
	class KP_Tracking_Parameter
	{
		protected $cookieName = 'kp_campaigns';
		protected $urlParameter = 'campaign';
		protected $cookieDuration = 604800; // 7 days in seconds

		/**
		 * Constructor.
		 * Hooks into WordPress to handle campaign tracking and exclude the cookie from WP Rocket cache.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.16
		 */
		public function __construct()
		{
			add_action('init', [$this, 'handleCampaignTracking']);
			add_filter('rocket_cache_dynamic_cookies', [$this, 'excludeCookieFromCache']);

			add_action('wp_enqueue_scripts', [$this, 'enqueueScripts']);
		}

		/**
		 * Enqueue the scripts.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.92
		 */
		public function enqueueScripts()
		{
			if (get_option('kls_toggle_anytime_integration') == 'on'){
				wp_enqueue_script('kls-anytime-joining', KP_ASSETS_PATH . 'build/public/anytime-joining.js', array('jquery'), KP_VERSION, true);
			}
		}

		/**
		 * Handles the tracking of campaign parameters.
		 * Checks for the 'campaign' URL parameter, updates the cookie value, and resets the cookie expiration.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.16
		 */
		public function handleCampaignTracking()
		{
			$campaignValue = $this->getCampaignValue();
			if (!$campaignValue) {
				return;
			}

			$currentCookieValue = isset($_COOKIE[$this->cookieName]) ? $_COOKIE[$this->cookieName] : '';
			$updatedCookieValue = $this->updateCookieValue($currentCookieValue, $campaignValue);

			setcookie($this->cookieName, $updatedCookieValue, time() + $this->cookieDuration, "/");
		}

		/**
		 * Retrieves the campaign value from the URL parameter 'campaign'.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.16
		 * 
		 * @return string|null The campaign parameter value, or null if not set.
		 */
		protected function getCampaignValue()
		{
			// Check for 'campaign' URL parameter
			if (isset($_GET[$this->urlParameter]) && !empty($_GET[$this->urlParameter])) {
				return $_GET[$this->urlParameter];
			}

			return null;
		}

		/**
		 * Updates the cookie value, appending new values without duplicating existing ones.
		 * 
		 * @param string $currentValue The current value of the cookie.
		 * @param string $newValue The new value to add to the cookie.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.16
		 * 
		 * @return string The updated cookie value.
		 */
		protected function updateCookieValue($currentValue, $newValue)
		{
			$values = array_filter(explode(',', $currentValue));
			if (empty($values) || end($values) !== $newValue) {
				$values[] = $newValue;
			}
			return implode(',', $values);
		}

		/**
		 * Adds the campaign cookie to the list of cookies excluded from WP Rocket's cache.
		 * 
		 * @param array $cookies The current array of cookies excluded from cache.
		 * 
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.1.16
		 * 
		 * @return array The updated array including the campaign cookie.
		 */
		public function excludeCookieFromCache($cookies)
		{
			$cookies[] = $this->cookieName;
			return $cookies;
		}
	}

	new KP_Tracking_Parameter();

}