<?php
/**
 * Get the current url from the website
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string Website url
 */
function kls_shortcode_current_url($atts) {
	$current_url = get_site_url();

	$strip_protocols  = array( 'https://', 'https://www.', 'http://', 'htpp://www.', 'www.' );
	$current_url_name = str_replace( $strip_protocols, '', $current_url );
	
	if (isset($atts['link']) && $atts['link'] == 0) {
		$html = $current_url . '/';
	} else {
		$html = '<a href="' . $current_url . '">' . $current_url_name . '</a>';
	}

	return $html;
}
add_shortcode( 'website_url', 'kls_shortcode_current_url' );

/**
 * Get the current year
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string The current year
 */
function kls_shortcode_current_year() {
	 return date( 'Y' );
}
add_shortcode( 'jaar', 'kls_shortcode_current_year' );

/**
 * Get the current page title
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string The title of the page
 */
function kls_title_shortcode() {
	return get_the_title();
}
add_shortcode( 'pagina_titel', 'kls_title_shortcode' );

/**
 * Cookie button text for the cookie statement page
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string Text of the cookie button
 */
function kls_shortcode_cookie_button_text() {
	return kls_plugin_translate_option( 'kls_cookie_button_text', 'Cookie notice button text', 'nl' );
}
add_shortcode( 'cookie_button_text', 'kls_shortcode_cookie_button_text' );

/**
 * Sitemap page shortcode
 *
 * @param array $atts An array of all the attributes put in the shortcode
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string List of all the posts, pages and portfolio items
 */
function kls_sitemap_shortcode( $atts ) {
	extract(
		shortcode_atts(
			array(
				'paginas'   => 1,
				'posts'     => 1,
				'projecten' => 0,
				'pagina_titel' => __( 'Pages', KP_TXT_DOMAIN ),
				'post_titel' => __('Posts', KP_TXT_DOMAIN),
				'project_titel' => __('Projects', KP_TXT_DOMAIN),
			),
			$atts
		)
	);

	wp_enqueue_style('kls_sitemap_css', KP_ASSETS_PATH . 'build/public/sitemap.css', false, KP_VERSION);

	ob_start();

	// Pages
	if ( $paginas == 1 ) :
		$page_args = array(
			'sort_order'   => 'asc',
			'sort_column'  => 'post_title',
			'hierarchical' => 1,
			'parent'       => -1,
			'offset'       => 0,
			'post_status'  => 'publish',
			'title_li'     => false,
		);
		?>
		
		<h2><?php echo $pagina_titel; ?></h2>
		<ul class="sitemap-list">
			<?php
			echo wp_list_pages( $page_args );
			?>
		</ul> 
		<?php
	endif;

	// Posts
	if ( $posts == 1 ) :
		$post_args = array(
			'numberposts' => '-1',
		);
		$posts     = get_posts($post_args);
		?>

		<h2><?php echo $post_titel; ?></h2>
		<ul class="sitemap-list">
			<?php
			foreach ( $posts as $post ) {
				$id    = $post->ID;
				$title = get_the_title( $id );
				$link  = get_permalink( $id );

				echo '<li><a href="' . $link . '">' . $title . '</a></li>';
			}
			?>
		</ul> 
		<?php
	endif;

	// Projects
	if ( $projecten == 1 ) :
		$project_args = array(
			'post_type'      => 'project',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'orderby'        => 'title',
			'order'          => 'ASC',
		);
		$projects     = new Wp_Query( $project_args );
		?>

		<h2><?php echo $project_titel; ?></h2>
		<ul class="sitemap-list">
			<?php
			// foreach($projects as $project){
			while ( $projects->have_posts() ) :
				$projects->the_post();
				$id    = $project->ID;
				$title = get_the_title( $id );
				$link  = get_permalink( $id );

				echo '<li><a href="' . $link . '">' . $title . '</a></li>';
			endwhile;
			// }
			?>
		</ul>

		<?php
	endif;

	return ob_get_clean();
}
add_shortcode( 'sitemap_pagina', 'kls_sitemap_shortcode' );

/**
 * Create spintax shortcode
 *
 * @param string $text An string of all the content put in the shortcode
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string Spinned text
 */
function kls_spintax( $text ) {
	 $spinner = new KlsSpintax();

	return $spinner->spin( $text );
}

/**
 * Create spintax shortcode
 *
 * @param array  $atts    An array of all the attributes put in the shortcode
 * @param string $content An string of all the content put in the shortcode
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 *
 * @return string Spinned text to shortcode
 */
function kls_render_shortcode( $atts, $content = '' ) {
	return do_shortcode( kls_spintax( $content ) );
}
add_shortcode( 'spintax', 'kls_render_shortcode' );

/**
 * Kls spintax class
 *
 * @author Joeri van der Stek <joeri@klassebv.nl>
 *
 * @since 0.0.79
 */
class KlsSpintax {

	/**
	 * Cleanup the text and spin it
	 *
	 * @param string $text An string of text put in by the shortcode
	 *
	 * @author Joeri van der Stek <joeri@klassebv.nl>
	 *
	 * @since 0.0.79
	 *
	 * @return string Text that has been spun
	 */
	public function spin( $text ) {
		 $text = trim( strval( $text ) );

		$spun = $this->do_spin( $text );

		return $spun;
	}

	/**
	 * Replace the text and run it through the spinner
	 *
	 * @param string $text An string of text put in by the shortcode
	 *
	 * @author Joeri van der Stek <joeri@klassebv.nl>
	 *
	 * @since 0.0.79
	 *
	 * @return string Spinned text
	 */
	private function do_spin( $text ) {
		return preg_replace_callback(
			'/\{(((?>[^\{\}]+)|(?R))*)\}/x',
			array( $this, 'replace' ),
			$text
		);
	}

	/**
	 * Replace the text of the specific spintax
	 *
	 * @param string $text An string of the specific type of
	 *
	 * @author Joeri van der Stek <joeri@klassebv.nl>
	 *
	 * @since 0.0.79
	 *
	 * @return string Spinned text
	 */
	private function replace( $text ) {
		 $text = $this->do_spin( $text[1] );
		$parts = explode( '|', $text );
		return $parts[ array_rand( $parts ) ];
	}
}
