<?php // phpcs:ignore
/**
 * Ninja Forms session class
 *
 * @category   Klasse_Plugin
 * @package    KP
 * @subpackage KP/public/includes/ninjaforms
 *
 * @author  Joeri van der Stek <joeri@klassebv.nl>
 * @license GPLV2 https://www.gnu.org/licenses/gpl-2.0.html
 * @link    https://klassebv.nl/
 * @since   0.0.94
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'KP_NF_Session' ) ) {
	/**
	 * Ninja Forms session class
	 *
	 * @category    Klasse_Plugin
	 * @package     KP
	 * @subpackage  KP/public/includes/ninjaforms
	 *
	 * @author  Joeri van der Stek <joeri@klassebv.nl>
	 * @license GPLV3 http://www.gnu.org/licenses/gpl-3.0.html
	 * @link    https://klassebv.nl
	 * @since   0.0.ninjaforms
	 */
	class KP_NF_Session {

		/**
		 * Construct the session.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 */
		public function __construct() {
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueueScripts' ) );

			add_action( 'ninja_forms_after_submission', array( $this, 'saveFormSubmissionToSession' ) );

			add_action( 'wp_ajax_kp_get_session_value', array( $this, 'getSessionValue' ) );
			add_action( 'wp_ajax_nopriv_kp_get_session_value', array( $this, 'getSessionValue' ) );

			add_shortcode( 'nf_session_value', array( $this, 'sessionShortcode' ) );
		}

		/**
		 * Enqueue the scripts.
		 *
		 * @author Joeri van der Stek <joeri@klassebv.nl>
		 *
		 * @since 0.0.94
		 */
		public function enqueueScripts() {
			wp_enqueue_script( 'nf-session-js', KP_ASSETS_PATH . 'build/public/nf-session.js', array( 'jquery' ), KP_VERSION, true );
			wp_localize_script( 'nf-session-js', 'kp_ajax_object', array( 'ajax_url' => admin_url( 'admin-ajax.php' ) ) );
		}

		/**
		 *
		 * Saves the form submission data to the WP_Session.
		 *
		 * This function saves the form submission data as an array of key-value pairs where each key corresponds to the field's unique identifier and each value corresponds to the field's submitted value.
		 *
		 * If the form field has options, it checks if the field's value matches one of the options. If it does, it saves the selected option's label and calculation value (if present) to the session.
		 *
		 * @author Joeri van der Stek joeri@klassebv.nl
		 *
		 * @since 0.0.94
		 *
		 * @param array $form_data The form submission data.
		 *
		 * @return array
		 */
		public function saveFormSubmissionToSession( $form_data ) {
			\Ninja_Forms()->session();

			if ( class_exists( 'WP_Session' ) ) {
				$wp_session = \WP_Session::get_instance();

				$fields = $form_data['fields'];

				foreach ( $fields as $field ) {
					$wp_session[ 'ninja_forms_submission_' . $field['key'] ] = $field['value'];

					// Add option information to the session.
					if ( isset( $field['options'] ) && is_array( $field['options'] ) ) {
						$matching_options = array_filter(
							$field['options'],
							function( $option ) use ( $field ) {
								return $field['value'] == $option['value'];
							}
						);
						$selected_option  = count( $matching_options ) == 1 ? array_pop( $matching_options ) : false;
					} else {
						$selected_option = false;
					}

					if ( $selected_option && isset( $selected_option['calc'] ) ) {
						$wp_session[ 'ninja_forms_submission_' . $field['key'] . ':calc' ] = $selected_option['calc'];
					}

					if ( $selected_option && isset( $selected_option['label'] ) ) {
						$wp_session[ 'ninja_forms_submission_' . $field['key'] . ':label' ] = $selected_option['label'];
					}
				}

				$wp_session->write_data();
				$wp_session->set_cookie();

				return $form_data;
			}
		}

		/**
		 * Retrieve the value of a session field and output it with optional value before and after.
		 *
		 * This function retrieves the value of a session field specified in the $_POST['field_key'] and outputs it along with the optional values set in $_POST['value_before'] and $_POST['value_after'].
		 * If the field key is not set or the session key is empty, the function will terminate.
		 *
		 * @author Joeri van der Stek joeri@klassebv.nl
		 *
		 * @since 0.0.94
		 */
		public function getSessionValue() {
			if ( isset( $_POST['nonce'] ) && ! wp_verify_nonce( $_POST['nonce'], 'get_session_value' ) ) {
				wp_die();
			}

			\Ninja_Forms()->session();
			if ( class_exists( 'WP_Session' ) ) {
				$wp_session  = WP_Session::get_instance();
				$field_key   = isset( $_POST['field_key'] ) ? wp_unslash( $_POST['field_key'] ) : '';
				$session_key = $wp_session[ 'ninja_forms_submission_' . $field_key ];
				$session_key = isset( $session_key ) ? $session_key : '';

				if ( $field_key && ! empty( $session_key ) ) {
					$value_before = isset( $_POST['value_before'] ) ? wp_unslash( $_POST['value_before'] ) : '';
					$value_after  = isset( $_POST['value_after'] ) ? wp_unslash( $_POST['value_after'] ) : '';
					echo esc_html( $value_before ) . esc_html( $session_key ) . esc_html( $value_after );
					wp_die();
				} else {
					wp_die();
				}
			}
		}

		/**
		 * Outputs the stored session value for a specific field key using a shortcode.
		 *
		 * The shortcode returns a span element containing the field value retrieved from the session, surrounded by the optional value_before and value_after parameters.
		 *
		 * @author Joeri van der Stek joeri@klassebv.nl
		 *
		 * @since 0.0.94
		 *
		 * @param array  $atts An associative array of attributes, used to define the field key and optional value before/after text.
		 * @param string $field_key The key of the field whose value should be retrieved from the session.
		 * @param string $value_before Text to be displayed before the retrieved field value.
		 * @param string $value_after Text to be displayed after the retrieved field value.
		 *
		 * @return string A span element with the class "kp-session-value" and data attributes containing the provided field key, value_before and value_after.
		 */
		public function sessionShortcode( $atts = array() ) {
			$field_key    = isset( $atts['field_key'] ) ? $atts['field_key'] : '';
			$value_before = isset( $atts['value_before'] ) ? $atts['value_before'] : '';
			$value_after  = isset( $atts['value_after'] ) ? $atts['value_after'] : '';
			$nonce        = wp_create_nonce( 'kp_get_session_value' );

			return '<span class="kp-session-value" data-field-key="' . esc_attr( $field_key ) . '" data-value-before="' . esc_html( $value_before ) . '" data-value-after="' . esc_html( $value_after ) . '" data-nonce="' . $nonce . '"></span>';
		}
	}

	new KP_NF_Session();
}
